<?php

namespace App\Http\Controllers;

use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class ProductController extends Controller
{
    // 1. Tampilkan Daftar Produk
    public function index()
    {
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Akses Ditolak');
        }

        $products = Product::latest()->get();
        return view('admin.products.index', compact('products'));
    }

    // 2. Tampilkan Form Tambah
    public function create()
    {
        if (Auth::user()->role !== 'admin') {
            abort(403);
        }
        return view('admin.products.create');
    }

    // 3. Simpan Produk Baru
    public function store(Request $request)
    {
        if (Auth::user()->role !== 'admin') {
            abort(403);
        }

        $request->validate([
            'name' => 'required',
            'price' => 'required|numeric',
            'stock' => 'required|numeric',
            'description' => 'nullable',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $imagePath = null;
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('products', 'public');
        }

        Product::create([
            'name' => $request->name,
            'price' => $request->price,
            'stock' => $request->stock,
            'description' => $request->description,
            'image' => $imagePath,
        ]);

        return redirect()->route('admin.products')->with('success', 'Produk berhasil ditambahkan.');
    }

    // 4. Tampilkan Form Edit
    public function edit($id)
    {
        if (Auth::user()->role !== 'admin') {
            abort(403);
        }
        $product = Product::findOrFail($id);
        return view('admin.products.edit', compact('product'));
    }

    // 5. Update Produk
    public function update(Request $request, $id)
    {
        if (Auth::user()->role !== 'admin') {
            abort(403);
        }

        $request->validate([
            'name' => 'required',
            'price' => 'required|numeric',
            'stock' => 'required|numeric',
            'description' => 'nullable',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $product = Product::findOrFail($id);

        if ($request->hasFile('image')) {
            // Hapus gambar lama jika ada
            if ($product->image) {
                Storage::delete('public/' . $product->image);
            }
            $product->image = $request->file('image')->store('products', 'public');
        }

        $product->update([
            'name' => $request->name,
            'price' => $request->price,
            'stock' => $request->stock,
            'description' => $request->description,
            'image' => $product->image, // Gunakan gambar baru atau tetap yang lama
        ]);

        return redirect()->route('admin.products')->with('success', 'Produk berhasil diperbarui.');
    }

    // 6. Hapus Produk (DENGAN TRY-CATCH SOLUSI 2)
    public function destroy($id)
    {
        if (Auth::user()->role !== 'admin') {
            abort(403);
        }

        try {
            $product = Product::findOrFail($id);

            // Coba hapus
            $product->delete();
            
            // Jika berhasil hapus database, hapus gambarnya juga (Opsional)
            if ($product->image) {
                Storage::delete('public/' . $product->image);
            }

            return redirect()->route('admin.products')->with('success', 'Produk telah dihapus.');

        } catch (\Illuminate\Database\QueryException $e) {
            // Cek kode error 1451 (Foreign Key Constraint)
            if ($e->errorInfo[1] == 1451) {
                return redirect()->back()->with('error', 'GAGAL HAPUS: Produk ini sudah pernah dibeli dan tercatat di riwayat pesanan. Tidak boleh dihapus demi keamanan data.');
            }
            // Error lain
            return redirect()->back()->with('error', 'Terjadi kesalahan database: ' . $e->getMessage());
        }
    }
}