<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB; // Untuk transaksi database
use App\Models\Cart;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Product;

class OrderController extends Controller
{
    // 1. Tampilkan Riwayat Pesanan
    public function index()
    {
        $orders = Order::where('user_id', Auth::id())
                        ->with('items.product') // Ambil detail item & nama produknya
                        ->latest()
                        ->get();

        return view('orders.index', compact('orders'));
    }

    // 2. Proses Checkout (JANTUNG APLIKASI)
    public function checkout()
    {
        $user = Auth::user();
        
        // Ambil semua barang di keranjang user
        $carts = Cart::where('user_id', $user->id)->get();

        if ($carts->isEmpty()) {
            return redirect()->back()->with('error', 'Keranjang Anda kosong.');
        }

        // Gunakan DB Transaction agar data aman (kalau gagal satu, gagal semua)
        DB::transaction(function () use ($user, $carts) {
            
            // A. Hitung Total Harga
            $totalPrice = $carts->sum(function($item) {
                return $item->product->price * $item->qty;
            });

            // B. Buat Data Order Baru
            $order = Order::create([
                'user_id' => $user->id,
                'total_price' => $totalPrice,
                'status' => 'Paid', // Anggap saja langsung lunas
            ]);

            // C. Pindahkan Isi Keranjang ke OrderItems & Kurangi Stok
            foreach ($carts as $cart) {
                OrderItem::create([
                    'order_id' => $order->id,
                    'product_id' => $cart->product_id,
                    'qty' => $cart->qty,
                    'price' => $cart->product->price, // Simpan harga saat beli
                ]);

                // Kurangi Stok Produk Asli
                $product = Product::find($cart->product_id);
                $product->decrement('stock', $cart->qty);
            }

            // D. Kosongkan Keranjang
            Cart::where('user_id', $user->id)->delete();
        });
        

        // Lempar ke halaman riwayat pesanan
        return redirect()->route('orders.index')->with('success', 'Pesanan berhasil dibuat! Terima kasih sudah berbelanja.');
    }

    // ... kode fungsi index dan checkout sebelumnya ...

    // --- FITUR BARU: ADMIN KELOLA PESANAN ---

    // 1. Tampilkan Semua Pesanan Masuk (Khusus Admin)
    public function indexAdmin()
    {
        // Cek apakah user adalah admin
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Hanya Admin yang boleh masuk sini!');
        }

        // Ambil semua order, urutkan dari yang terbaru
        $orders = Order::with('user')->latest()->get();

        return view('admin.orders.index', compact('orders'));
    }

    // 2. Update Status Pesanan (Misal: Dikirim / Selesai)
    public function updateStatus(Request $request, $id)
    {
        if (Auth::user()->role !== 'admin') {
            abort(403);
        }

        $order = Order::findOrFail($id);
        
        // Validasi input status
        $request->validate([
            'status' => 'required|in:Unpaid,Paid,Shipped,Completed,Cancelled'
        ]);

        $order->update(['status' => $request->status]);

        return redirect()->back()->with('success', 'Status pesanan berhasil diperbarui!');
    }
}