<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Models\Cart;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Product;

class OrderController extends Controller
{
    // ==========================================
    // BAGIAN UNTUK PEMBELI (USER)
    // ==========================================

    // 1. Tampilkan Riwayat Pesanan
    public function index()
    {
        $orders = Order::where('user_id', Auth::id())
                        ->with('items.product') 
                        ->latest()
                        ->get();

        return view('orders.index', compact('orders'));
    }

    // 2. Proses Checkout
    public function checkout()
    {
        $user = Auth::user();
        $carts = Cart::where('user_id', $user->id)->get();

        if ($carts->isEmpty()) {
            return redirect()->back()->with('error', 'Keranjang Anda kosong.');
        }

        DB::transaction(function () use ($user, $carts) {
            
            // Hitung Total
            $totalPrice = $carts->sum(function($item) {
                return $item->product->price * $item->qty;
            });

            // Buat Order (PERBAIKAN 1: Status jadi huruf kecil 'paid')
            $order = Order::create([
                'user_id' => $user->id,
                'total_price' => $totalPrice,
                'status' => 'paid', 
            ]);

            // Pindahkan Item
            foreach ($carts as $cart) {
                OrderItem::create([
                    'order_id' => $order->id,
                    'product_id' => $cart->product_id,
                    'qty' => $cart->qty,
                    'price' => $cart->product->price,
                ]);

                // Kurangi Stok
                $product = Product::find($cart->product_id);
                if($product) {
                    $product->decrement('stock', $cart->qty);
                }
            }

            // Hapus Keranjang
            Cart::where('user_id', $user->id)->delete();
        });

        return redirect()->route('orders.index')->with('success', 'Pesanan berhasil dibuat!');
    }

    // ==========================================
    // BAGIAN UNTUK ADMIN (KELOLA PESANAN)
    // ==========================================

    // 3. Tampilkan Semua Pesanan Masuk
    public function indexAdmin()
    {
        if (Auth::user()->role !== 'admin') {
            abort(403, 'Hanya Admin yang boleh masuk sini!');
        }

        $orders = Order::with('user')->latest()->get();

        // Pastikan view-nya ada di folder resources/views/admin/orders/index.blade.php
        return view('admin.orders.index', compact('orders'));
    }

    // 4. Update Status Pesanan
    public function updateStatus(Request $request, $id)
    {
        // Cek admin lagi untuk keamanan
        if (Auth::user()->role !== 'admin') {
            abort(403);
        }

        $order = Order::findOrFail($id);
        
        // PERBAIKAN 2: Validasi disesuaikan dengan value di Blade (huruf kecil semua)
        $request->validate([
            'status' => 'required|in:pending,paid,shipped,completed,cancelled'
        ]);

        // Update database
        $order->update(['status' => $request->status]);

        return redirect()->back()->with('success', 'Status pesanan berhasil diperbarui!');
    }
}